<?php
session_start();
require_once '../includes/config.php';
require_once '../includes/functions.php';

if (!isLoggedIn()) {
    redirectToLogin();
}

$message = '';
$error = '';
$blog = [
    'id' => '',
    'title' => '',
    'slug' => '',
    'category' => '',
    'image' => '',
    'content' => '',
    'status' => 'draft'
];

if (isset($_GET['id'])) {
    $id = (int)$_GET['id'];
    $fetchedBlog = getBlogById($id);
    if ($fetchedBlog) {
        $blog = $fetchedBlog;
    }
}

// Handle Form Submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $id = $_POST['id'] ?? '';
    $title = sanitizeInput($_POST['title']);
    $slug = sanitizeInput($_POST['slug']);
    if (empty($slug)) $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $title)));
    $category = sanitizeInput($_POST['category']);
    $status = $_POST['status'];
    $content = $_POST['content']; // Don't sanitize HTML content for editor
    $image = $_POST['current_image'] ?? '';

    // Handle Image Upload
    if (isset($_FILES['image']) && $_FILES['image']['error'] === 0) {
        $uploadDir = '../assets/images/blog/';
        if (!file_exists($uploadDir)) mkdir($uploadDir, 0777, true);
        
        $fileName = time() . '_' . basename($_FILES['image']['name']);
        $targetFile = $uploadDir . $fileName;
        
        if (move_uploaded_file($_FILES['image']['tmp_name'], $targetFile)) {
            $image = 'assets/images/blog/' . $fileName;
        }
    }

    try {
        if ($id) {
            // Update
            $stmt = $conn->prepare("UPDATE blogs SET title=?, slug=?, category=?, content=?, image=?, status=? WHERE id=?");
            $stmt->execute([$title, $slug, $category, $content, $image, $status, $id]);
            $message = "Blog updated successfully!";
        } else {
            // Insert
            $stmt = $conn->prepare("INSERT INTO blogs (title, slug, category, content, image, status) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([$title, $slug, $category, $content, $image, $status]);
            $id = $conn->lastInsertId();
            $message = "Blog created successfully!";
        }
        
        // Refresh data
        $fetchedBlog = getBlogById($id);
        if ($fetchedBlog) $blog = $fetchedBlog;
        
    } catch(PDOException $e) {
        $error = "Error saving blog: " . $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $blog['id'] ? 'Edit' : 'New'; ?> Blog - Admin</title>
    
    <?php include '../includes/favicon-meta.php'; ?>
    
    <link rel="stylesheet" href="../assets/css/style.css?v=<?php echo time(); ?>">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <!-- CKEditor 5 -->
    <script src="https://cdn.ckeditor.com/ckeditor5/39.0.1/classic/ckeditor.js"></script>
    <style>
        body, html, a, button, input, textarea { cursor: auto !important; }
        .ck-editor__editable_inline {
            min-height: 400px;
            background-color: var(--dark-bg) !important;
            color: var(--text-primary) !important;
        }
        .ck.ck-editor__main>.ck-editor__editable:not(.ck-focused) {
            border-color: var(--border-color) !important;
        }
        .ck.ck-toolbar {
            background-color: var(--dark-card) !important;
            border-color: var(--border-color) !important;
        }
        .ck.ck-button {
            color: var(--text-primary) !important;
        }
        .ck.ck-button:hover {
            background: var(--dark-hover) !important;
        }
    </style>
    <style>
        .admin-layout { display: flex; min-height: 100vh; }
        .admin-sidebar { width: 260px; background: var(--dark-card); border-right: 1px solid var(--border-color); padding: var(--spacing-md); }
        .admin-content { flex: 1; padding: var(--spacing-lg); }
        .form-group { margin-bottom: 20px; }
        .form-label { display: block; margin-bottom: 8px; color: var(--text-primary); font-weight: 600; }
        .form-control { width: 100%; padding: 12px; background: var(--dark-bg); border: 1px solid var(--border-color); color: white; border-radius: 8px; }
        .form-control:focus { outline: none; border-color: var(--primary-color); }
        .btn-submit { background: var(--primary-gradient); border: none; padding: 12px 30px; color: white; border-radius: 8px; cursor: pointer; font-weight: 600; }
    </style>
</head>
<body>
    <div class="admin-layout">
        <aside class="admin-sidebar">
            <div class="logo" style="margin-bottom: var(--spacing-lg); font-size: 24px; font-weight: 800; color: var(--primary-color);">LinkBoostify</div>
            <nav style="display: flex; flex-direction: column; gap: 10px;">
                <a href="<?php echo SITE_URL . '/' . ADMIN_PATH; ?>/index.php" style="color: var(--text-secondary); padding: 10px;"><i class="fas fa-home"></i> Dashboard</a>
                <a href="<?php echo SITE_URL . '/' . ADMIN_PATH; ?>/blogs.php" style="color: var(--primary-color); background: var(--dark-bg); padding: 10px; border-radius: 8px;"><i class="fas fa-blog"></i> Blogs</a>
                <a href="<?php echo SITE_URL . '/' . ADMIN_PATH; ?>/contacts.php" style="color: var(--text-secondary); padding: 10px;"><i class="fas fa-envelope"></i> Messages</a>
                <a href="<?php echo SITE_URL . '/' . ADMIN_PATH; ?>/settings.php" style="color: var(--text-secondary); padding: 10px;"><i class="fas fa-cog"></i> Settings</a>
                <a href="<?php echo SITE_URL . '/' . ADMIN_PATH; ?>/logout.php" style="color: #ff4757; padding: 10px; margin-top: 20px;"><i class="fas fa-sign-out-alt"></i> Logout</a>
            </nav>
        </aside>
        
        <main class="admin-content">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px;">
                <h1><?php echo $blog['id'] ? 'Edit' : 'New'; ?> Blog Post</h1>
                <a href="blogs.php" class="btn btn-outline"><i class="fas fa-arrow-left"></i> Back</a>
            </div>

            <?php if($message): ?><div style="padding: 15px; background: rgba(46, 213, 115, 0.1); color: #2ed573; border-radius: 8px; margin-bottom: 20px;"><?php echo $message; ?></div><?php endif; ?>
            <?php if($error): ?><div style="padding: 15px; background: rgba(255, 71, 87, 0.1); color: #ff4757; border-radius: 8px; margin-bottom: 20px;"><?php echo $error; ?></div><?php endif; ?>
            
            <form method="POST" enctype="multipart/form-data" style="background: var(--dark-card); padding: 30px; border-radius: 12px; border: 1px solid var(--border-color);">
                <input type="hidden" name="id" value="<?php echo $blog['id']; ?>">
                <input type="hidden" name="current_image" value="<?php echo $blog['image']; ?>">
                
                <div class="form-group">
                    <label class="form-label">Title</label>
                    <input type="text" name="title" class="form-control" value="<?php echo htmlspecialchars($blog['title']); ?>" required>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                    <div class="form-group">
                        <label class="form-label">Slug (URL)</label>
                        <input type="text" name="slug" class="form-control" value="<?php echo htmlspecialchars($blog['slug']); ?>" placeholder="Auto-generated from title">
                    </div>
                    <div class="form-group">
                        <label class="form-label">Category</label>
                        <select name="category" class="form-control">
                            <option value="SEO" <?php echo $blog['category'] == 'SEO' ? 'selected' : ''; ?>>SEO</option>
                            <option value="Link Building" <?php echo $blog['category'] == 'Link Building' ? 'selected' : ''; ?>>Link Building</option>
                            <option value="Digital Marketing" <?php echo $blog['category'] == 'Digital Marketing' ? 'selected' : ''; ?>>Digital Marketing</option>
                            <option value="Content Strategy" <?php echo $blog['category'] == 'Content Strategy' ? 'selected' : ''; ?>>Content Strategy</option>
                        </select>
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">Featured Image</label>
                    <input type="file" name="image" class="form-control" accept="image/*">
                    <?php if($blog['image']): ?>
                        <div style="margin-top: 10px;">
                            <img src="../<?php echo $blog['image']; ?>" alt="Current" style="height: 100px; border-radius: 8px;">
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Content</label>
                    <textarea name="content" id="content" rows="20"><?php echo $blog['content']; ?></textarea>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Status</label>
                    <select name="status" class="form-control" style="width: 200px;">
                        <option value="published" <?php echo $blog['status'] == 'published' ? 'selected' : ''; ?>>Published</option>
                        <option value="draft" <?php echo $blog['status'] == 'draft' ? 'selected' : ''; ?>>Draft</option>
                    </select>
                </div>
                
                <button type="submit" class="btn-submit">
                    <i class="fas fa-save"></i> Save Blog Post
                </button>
            </form>
        </main>
    </div>
    <script src="../assets/js/main.js"></script>
    <script>
        ClassicEditor
            .create(document.querySelector('#content'))
            .then(editor => {
                console.log('Editor was initialized', editor);
            })
            .catch(error => {
                console.error(error);
            });
    </script>
</body>
</html>
