// LinkBoostify - Main JavaScript

// Header Scroll Effect
// Header Scroll Effect
window.addEventListener('scroll', () => {
    const header = document.querySelector('.header');
    if (header) {
        if (window.scrollY > 50) {
            header.classList.add('scrolled');
        } else {
            header.classList.remove('scrolled');
        }
    }
});

// Counter Animation
function animateCounter(element) {
    const target = parseInt(element.getAttribute('data-count'));
    const duration = 2000;
    const step = target / (duration / 16);
    let current = 0;

    const timer = setInterval(() => {
        current += step;
        if (current >= target) {
            element.textContent = target + '+';
            clearInterval(timer);
        } else {
            element.textContent = Math.floor(current);
        }
    }, 16);
}

// Intersection Observer for Counter Animation
const observerOptions = {
    threshold: 0.5,
    rootMargin: '0px'
};

const observer = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
        if (entry.isIntersecting) {
            const counters = entry.target.querySelectorAll('.stat-number');
            counters.forEach(counter => {
                if (!counter.classList.contains('animated')) {
                    counter.classList.add('animated');
                    animateCounter(counter);
                }
            });
        }
    });
}, observerOptions);

// Observe stats section
const statsSection = document.querySelector('.stats-section');
if (statsSection) {
    observer.observe(statsSection);
}

// Mobile Menu Toggle
const mobileMenuToggle = document.querySelector('.mobile-menu-toggle');
const navMenu = document.querySelector('.nav-menu');

if (mobileMenuToggle) {
    mobileMenuToggle.addEventListener('click', () => {
        navMenu.classList.toggle('active');
        const icon = mobileMenuToggle.querySelector('i');
        if (navMenu.classList.contains('active')) {
            icon.classList.remove('fa-bars');
            icon.classList.add('fa-times');
        } else {
            icon.classList.remove('fa-times');
            icon.classList.add('fa-bars');
        }
    });
}

// Mobile Dropdown Toggle
document.addEventListener('DOMContentLoaded', () => {
    const mobileDropdowns = document.querySelectorAll('.nav-menu .dropdown');

    mobileDropdowns.forEach(dropdown => {
        const dropdownLink = dropdown.querySelector('.nav-link');
        const dropdownMenu = dropdown.querySelector('.dropdown-menu');

        if (dropdownLink && dropdownMenu) {
            dropdownLink.addEventListener('click', (e) => {
                // Only toggle on mobile (when menu is in mobile mode)
                if (window.innerWidth <= 992) {
                    e.preventDefault();
                    dropdown.classList.toggle('active');
                }
            });
        }
    });
});

// Scroll Reveal Animation (Global)
window.addEventListener('load', () => {
    const revealObserverOptions = {
        threshold: 0.01, // Trigger as soon as 1% is visible (Very Sensitive)
        rootMargin: "0px 0px 0px 0px" // No negative margins, trigger immediately on enter
    };

    const revealObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('active');
                observer.unobserve(entry.target);
            }
        });
    }, revealObserverOptions);

    // Observe all elements with .reveal class and new variants
    const revealElements = document.querySelectorAll('.reveal, .reveal-wipe, .reveal-left, .reveal-right, .reveal-scale');
    revealElements.forEach(el => {
        revealObserver.observe(el);
    });

    // Observe all elements with .reveal-wipe class
    const wipeElements = document.querySelectorAll('.reveal-wipe');
    wipeElements.forEach(el => {
        revealObserver.observe(el);
    });



    console.log('Animation Observers Initialized after Load');
});


// Smooth Scroll for Anchor Links
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
        const href = this.getAttribute('href');
        if (href !== '#') {
            e.preventDefault();
            const target = document.querySelector(href);
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        }
    });
});

// Form Validation
const contactForm = document.querySelector('#contact-form');
if (contactForm) {
    contactForm.addEventListener('submit', function (e) {
        e.preventDefault();

        const name = this.querySelector('[name="name"]').value.trim();
        const email = this.querySelector('[name="email"]').value.trim();
        const subject = this.querySelector('[name="subject"]').value.trim();
        const message = this.querySelector('[name="message"]').value.trim();

        if (!name || !email || !subject || !message) {
            showNotification('Please fill in all fields', 'error');
            return;
        }

        if (!isValidEmail(email)) {
            showNotification('Please enter a valid email address', 'error');
            return;
        }

        // Submit form
        this.submit();
    });
}

// Email Validation
function isValidEmail(email) {
    const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return re.test(email);
}

// Notification System
function showNotification(message, type = 'success') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;

    document.body.appendChild(notification);

    setTimeout(() => {
        notification.classList.add('show');
    }, 100);

    setTimeout(() => {
        notification.classList.remove('show');
        setTimeout(() => {
            notification.remove();
        }, 300);
    }, 3000);
}

// Add notification styles dynamically
const notificationStyles = `
    .notification {
        position: fixed;
        top: 100px;
        right: 20px;
        padding: 16px 24px;
        background: var(--dark-card);
        border: 1px solid var(--border-color);
        border-radius: var(--radius-md);
        color: var(--text-primary);
        box-shadow: var(--shadow-lg);
        transform: translateX(400px);
        transition: transform 0.3s ease;
        z-index: 10000;
        max-width: 300px;
    }
    
    .notification.show {
        transform: translateX(0);
    }
    
    .notification-success {
        border-left: 4px solid #43e97b;
    }
    
    .notification-error {
        border-left: 4px solid #f5576c;
    }
`;

const styleSheet = document.createElement('style');
styleSheet.textContent = notificationStyles;
document.head.appendChild(styleSheet);

// Lazy Loading Images
if ('IntersectionObserver' in window) {
    const imageObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const img = entry.target;
                img.src = img.dataset.src;
                img.classList.remove('lazy');
                imageObserver.unobserve(img);
            }
        });
    });

    document.querySelectorAll('img.lazy').forEach(img => {
        imageObserver.observe(img);
    });
}

// Page Load Animation
window.addEventListener('load', () => {
    document.body.classList.add('loaded');
});

// Custom Cursor Logic
document.addEventListener('DOMContentLoaded', () => {
    // Check if device is desktop (touch devices don't need custom cursor)
    if (window.matchMedia("(min-width: 992px)").matches) {

        // Create cursor elements
        const cursorDot = document.createElement('div');
        cursorDot.classList.add('cursor-dot');

        const cursorOutline = document.createElement('div');
        cursorOutline.classList.add('cursor-outline');

        const cursorSpotlight = document.createElement('div');
        cursorSpotlight.classList.add('cursor-spotlight');

        document.body.classList.add('custom-cursor-active'); // Enable CSS cursor hiding

        document.body.appendChild(cursorSpotlight); // Append spotlight first so it's behind
        document.body.appendChild(cursorDot);
        document.body.appendChild(cursorOutline);

        // Track mouse movement
        window.addEventListener('mousemove', (e) => {
            const posX = e.clientX;
            const posY = e.clientY;

            // Dot follows instantly
            cursorDot.style.left = `${posX}px`;
            cursorDot.style.top = `${posY}px`;

            // Outline follows with slight delay
            cursorOutline.animate({
                left: `${posX}px`,
                top: `${posY}px`
            }, { duration: 500, fill: "forwards" });

            // Spotlight follows with slight delay
            cursorSpotlight.animate({
                left: `${posX}px`,
                top: `${posY}px`
            }, { duration: 500, fill: "forwards" });
        });

        // Hero Spotlight Effect (Only on Heading)
        const heroHeading = document.querySelector('.hero-title');
        if (heroHeading) {
            heroHeading.addEventListener('mouseenter', () => {
                document.body.classList.add('in-hero');
            });
            heroHeading.addEventListener('mouseleave', () => {
                document.body.classList.remove('in-hero');
            });
        }


    }
});

// ===================================
// Scroll Reveal Animation for Images
// ===================================

function initScrollReveal() {
    const revealElements = document.querySelectorAll('.image-anime, .reveal-anime');

    const revealObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('reveal');
                revealObserver.unobserve(entry.target);
            }
        });
    }, {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    });

    revealElements.forEach(element => {
        revealObserver.observe(element);
    });
}


// Typewriter / Word Reveal Animation
function initTypeReveal() {
    const elements = document.querySelectorAll('.type-reveal, .type-reveal-word');

    elements.forEach((el, elIndex) => {
        // Helper to process nodes recursively
        function processNodes(parent) {
            const nodes = Array.from(parent.childNodes);
            parent.innerHTML = ''; // Clear content

            nodes.forEach(node => {
                if (node.nodeType === Node.TEXT_NODE) {
                    const text = node.textContent;
                    if (!text.trim()) {
                        // Preserve spaces if needed, or just ignore empty text nodes
                        if (text.includes(' ')) parent.appendChild(document.createTextNode(' '));
                        return;
                    }

                    const words = text.split(/(\s+)/); // Split keep delimiters
                    words.forEach(word => {
                        if (word.trim()) {
                            const span = document.createElement('span');
                            span.textContent = word;
                            span.className = 'word-reveal-span';
                            span.style.opacity = '0';
                            span.style.transform = 'translateY(10px)';
                            span.style.display = 'inline-block';
                            span.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
                            parent.appendChild(span);
                        } else {
                            // It's whitespace
                            parent.appendChild(document.createTextNode(word));
                        }
                    });
                } else if (node.nodeType === Node.ELEMENT_NODE) {
                    // Create a new container for the element's processed children
                    const newElement = node.cloneNode(false); // Clone without children
                    processNodes(node); // Process children of the original node
                    // Move children from original node to newElement
                    while (node.firstChild) {
                        newElement.appendChild(node.firstChild);
                    }
                    parent.appendChild(newElement); // Append the new element with processed children
                }
            });
        }

        // Store original content to restore if needed, though not strictly necessary for this animation
        // const originalContent = el.innerHTML; 

        processNodes(el);

        // Observer to trigger reveal with stagger
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const spans = entry.target.querySelectorAll('.word-reveal-span');
                    spans.forEach((span, index) => {
                        setTimeout(() => {
                            span.style.opacity = '1';
                            span.style.transform = 'translateY(0)';
                        }, index * 50); // 50ms stagger
                    });
                    observer.unobserve(entry.target);
                }
            });
        }, { threshold: 0.1 });

        observer.observe(el);
    });
}


// Initialize on DOM load
document.addEventListener('DOMContentLoaded', () => {
    initScrollReveal();
    initTypeReveal();
});

console.log('🚀 LinkBoostify - Digital Marketing Agency');
