<?php
session_start();
require_once 'includes/config.php';
require_once 'includes/functions.php';

$success = false;
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = sanitizeInput($_POST['name'] ?? '');
    $email = sanitizeInput($_POST['email'] ?? '');
    $subject = sanitizeInput($_POST['subject'] ?? '');
    $message = sanitizeInput($_POST['message'] ?? '');
    
    if (empty($name) || empty($email) || empty($subject) || empty($message)) {
        $error = 'Please fill in all fields.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Please enter a valid email address.';
    } else {
        try {
            // Save to Database
            $stmt = $conn->prepare("INSERT INTO contact_submissions (name, email, subject, message) VALUES (?, ?, ?, ?)");
            $stmt->execute([$name, $email, $subject, $message]);
            
            // Optionally send email
            sendContactEmail($name, $email, $subject, $message);
            
            $success = true;
        } catch(PDOException $e) {
            $error = 'Something went wrong. Please try again.';
        }
    }
}

$page_title = "Contact Us - LinkBoostify";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?></title>
    
    <?php include 'includes/favicon-meta.php'; ?>
    
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        .contact-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: var(--spacing-xl);
            margin-top: var(--spacing-xl);
        }
        
        .contact-info-card {
            padding: var(--spacing-lg);
            background: var(--dark-card);
            border: 1px solid var(--border-color);
            border-radius: var(--radius-lg);
            text-align: center;
        }
        
        .contact-info-icon {
            width: 64px;
            height: 64px;
            margin: 0 auto var(--spacing-md);
            display: flex;
            align-items: center;
            justify-content: center;
            background: var(--primary-gradient);
            border-radius: var(--radius-md);
            font-size: 28px;
        }
        
        .form-group {
            margin-bottom: var(--spacing-md);
        }
        
        .form-label {
            display: block;
            margin-bottom: var(--spacing-xs);
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .form-control {
            width: 100%;
            padding: 14px 20px;
            background: var(--dark-card);
            border: 1px solid var(--border-color);
            border-radius: var(--radius-md);
            color: var(--text-primary);
            font-family: var(--font-primary);
            font-size: 16px;
            transition: var(--transition-base);
        }
        
        .form-control:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        textarea.form-control {
            min-height: 150px;
            resize: vertical;
        }
        
        .alert {
            padding: var(--spacing-md);
            border-radius: var(--radius-md);
            margin-bottom: var(--spacing-md);
        }
        
        .alert-success {
            background: rgba(67, 233, 123, 0.1);
            border: 1px solid #43e97b;
            color: #43e97b;
        }
        
        .alert-error {
            background: rgba(245, 87, 108, 0.1);
            border: 1px solid #f5576c;
            color: #f5576c;
        }
        
        @media (max-width: 968px) {
            .contact-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <section class="page-header" style="padding: 120px 0 60px; background: var(--dark-card); margin-top: 70px;">
        <div class="container">
            <div class="section-header">
                <span class="section-tag">Get In Touch</span>
                <h1 class="section-title">Contact Us</h1>
                <p style="color: var(--text-secondary); max-width: 600px; margin: 0 auto;">
                    Let's achieve social media excellence together. Reach out to us today!
                </p>
            </div>
        </div>
    </section>

    <section style="padding: var(--spacing-xxl) 0;">
        <div class="container">
            <div class="contact-grid">
                <div>
                    <h2 style="margin-bottom: var(--spacing-md);">Send us a message</h2>
                    
                    <?php if ($success): ?>
                        <div class="alert alert-success">
                            <i class="fas fa-check-circle"></i> Thank you! Your message has been sent successfully.
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($error): ?>
                        <div class="alert alert-error">
                            <i class="fas fa-exclamation-circle"></i> <?php echo $error; ?>
                        </div>
                    <?php endif; ?>
                    
                    <form method="POST" id="contact-form">
                        <div class="form-group">
                            <label class="form-label">Your Name</label>
                            <input type="text" name="name" class="form-control" required>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Your Email</label>
                            <input type="email" name="email" class="form-control" required>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Subject</label>
                            <input type="text" name="subject" class="form-control" required>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Message</label>
                            <textarea name="message" class="form-control" required></textarea>
                        </div>
                        
                        <button type="submit" class="btn btn-primary btn-large" style="width: 100%;">
                            Send Message <i class="fas fa-paper-plane"></i>
                        </button>
                    </form>
                </div>
                
                <div>
                    <h2 style="margin-bottom: var(--spacing-md);">Contact Information</h2>
                    
                    <div style="display: flex; flex-direction: column; gap: var(--spacing-md);">
                        <div class="contact-info-card">
                            <div class="contact-info-icon">
                                <i class="fas fa-phone"></i>
                            </div>
                            <h3>Phone</h3>
                            <p style="color: var(--text-secondary); margin-top: 8px;">
                                <a href="tel:<?php echo SITE_PHONE; ?>" style="color: var(--primary-light);">
                                    <?php echo SITE_PHONE; ?>
                                </a>
                            </p>
                        </div>
                        
                        <div class="contact-info-card">
                            <div class="contact-info-icon">
                                <i class="fas fa-envelope"></i>
                            </div>
                            <h3>Email</h3>
                            <p style="color: var(--text-secondary); margin-top: 8px;">
                                <a href="mailto:linkboostify@gmail.com" style="color: var(--primary-light);">
                                    linkboostify@gmail.com
                                </a>
                            </p>
                        </div>
                        
                        <div class="contact-info-card">
                            <div class="contact-info-icon">
                                <i class="fab fa-whatsapp"></i>
                            </div>
                            <h3>WhatsApp</h3>
                            <p style="color: var(--text-secondary); margin-top: 8px;">
                                <a href="https://api.whatsapp.com/send?phone=<?php echo str_replace('+', '', SITE_WHATSAPP); ?>" target="_blank" style="color: var(--primary-light);">
                                    Chat with us
                                </a>
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <?php include 'includes/footer.php'; ?>

    <script src="assets/js/main.js"></script>
</body>
</html>
